// V5 Dashboard Map (Sheet-connected)
const API_URL = "./proxy.php";
const REFRESH_MS = 5000;

const COL_NAME = "Name";
const COL_ADDRESS = "Address";
const COL_PHONE = "Phone";
const COL_WEBSITE = "Website";
const COL_MAPS_URL = "Google Maps URL";
const COL_PLACE_ID = "Place ID";
const COL_LABELS = "Labels";
const COL_NOTES = "Notes";
const COL_RATING = "Rating";
const COL_REVIEWS = "Reviews";
const COL_AREA = "Area";
const COL_NEEDS_WEBSITE = "Needs Website";
const COL_LAT = "Lat";
const COL_LNG = "Lng";

const LABELS_CORE = [
  "Unsent","Sent","Seen","Not replied",
  "Replied","Interested","Not Interested","Follow-up","Closed"
];

let map, markersLayer;
let allLeads = [];
let markersById = new Map();
let lastRev = null;

let activeSearch = "";
let activeLabel = "";
let activeNeedsWebsite = "";
let activeChip = "";

function $(id){return document.getElementById(id);}

function parseLabels(str){
  return (str||"").toString().split(",").map(s=>s.trim()).filter(Boolean);
}
function hasLabel(lead, lbl){
  return parseLabels(lead.labels).includes(lbl);
}
function escapeHtml(s){
  return (s ?? "").toString().replaceAll("&","&amp;").replaceAll("<","&lt;").replaceAll(">","&gt;");
}
function escapeAttr(s){
  return (s ?? "").toString().replaceAll('"',"&quot;");
}

async function fetchSheet(){
  const res = await fetch(API_URL, {cache:"no-store"});
  const json = await res.json();
  if(!json.ok) throw new Error(json.error || "Sheet API error");
  return json;
}

async function updateRow(id, labels, notes){
  const res = await fetch(API_URL,{
    method:"POST",
    headers:{"Content-Type":"application/json"},
    body: JSON.stringify({id, labels, notes})
  });
  const json = await res.json();
  if(!json.ok) throw new Error(json.error || "Update failed");
  return json;
}

function initMap(){
  map = L.map("map").setView([25.7617, -80.1918], 11);
  L.tileLayer("https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png",{
    maxZoom:19, attribution:"&copy; OpenStreetMap"
  }).addTo(map);
  markersLayer = L.layerGroup().addTo(map);
}

function rebuildLabelDropdown(){
  const sel = $("labelFilter");
  const set = new Set();
  for(const l of allLeads){
    parseLabels(l.labels).forEach(x=>set.add(x));
  }
  const list = Array.from(set).sort((a,b)=>a.localeCompare(b));
  sel.innerHTML = `<option value="">All labels</option>` +
    list.map(x=>`<option value="${escapeAttr(x)}">${escapeHtml(x)}</option>`).join("");
}

function leadMatches(lead){
  const s = activeSearch.trim().toLowerCase();
  if(s){
    const hay = [lead.name, lead.address, lead.area, lead.phone, lead.website].join(" ").toLowerCase();
    if(!hay.includes(s)) return false;
  }
  if(activeNeedsWebsite){
    if((lead.needsWebsite||"").toString().toUpperCase() !== activeNeedsWebsite) return false;
  }
  if(activeLabel){
    if(!hasLabel(lead, activeLabel)) return false;
  }
  if(activeChip){
    if(activeChip === "Unsent"){
      if(hasLabel(lead,"Sent")) return false;
    }else{
      if(!hasLabel(lead, activeChip)) return false;
    }
  }
  return true;
}

function computeStats(){
  const stat = {Unsent:0, Sent:0, Seen:0, "Not replied":0};
  for(const l of allLeads){
    const labels = parseLabels(l.labels);
    if(!labels.includes("Sent")) stat.Unsent++;
    if(labels.includes("Sent")) stat.Sent++;
    if(labels.includes("Seen")) stat.Seen++;
    if(labels.includes("Not replied")) stat["Not replied"]++;
  }
  $("statUnsent").textContent = stat.Unsent;
  $("statSent").textContent = stat.Sent;
  $("statSeen").textContent = stat.Seen;
  $("statNotReplied").textContent = stat["Not replied"];
}

function renderList(leads){
  const listEl = $("list");
  listEl.innerHTML = "";

  leads.forEach(lead=>{
    const div = document.createElement("div");
    div.className = "lead";
    div.dataset.id = lead.id;

    div.innerHTML = `
      <div class="top">
        <div style="min-width:0;">
          <h3 title="${escapeAttr(lead.name)}">${escapeHtml(lead.name || "(no name)")}</h3>
          <div class="meta">${escapeHtml(lead.address || "")}</div>
          <div class="meta">${escapeHtml(lead.area || "")}</div>
        </div>
        <div style="text-align:right; min-width:90px;">
          <div style="font-weight:800; font-size:12px;">${escapeHtml(lead.rating ? lead.rating + "⭐" : "")}</div>
          <div style="color:rgba(234,240,255,.55); font-size:11px;">${escapeHtml(lead.reviews ? lead.reviews + " reviews" : "")}</div>
        </div>
      </div>

      <div style="display:flex; gap:8px; flex-wrap:wrap; margin-top:10px;">
        <button class="qbtn primary" data-label="Sent">Sent</button>
        <button class="qbtn" data-label="Seen">Seen</button>
        <button class="qbtn" data-label="Not replied">Not replied</button>
        <button class="qbtn" data-label="Offer">Offer</button>
        <button class="qbtn" data-label="Follow-up">Follow-up</button>
        <button class="qbtn danger" data-label="Not Interested">Not Interested</button>
      </div>
    `;

    div.addEventListener("click", ()=>{
      focusLead(lead.id, true);
    });

    div.querySelectorAll("button.qbtn").forEach(btn=>{
      btn.addEventListener("click", async (e)=>{
        e.stopPropagation();
        await quickAddLabel(lead.id, btn.dataset.label);
      });
    });

    listEl.appendChild(div);
  });
}

function renderMarkers(leads){
  markersLayer.clearLayers();
  markersById.clear();

  for(const lead of leads){
    if(!lead.lat || !lead.lng) continue;

    const marker = L.circleMarker([lead.lat, lead.lng],{
      radius:8,
      color:"rgba(76,141,255,.95)",
      weight:2,
      fillColor:"rgba(76,141,255,.35)",
      fillOpacity:0.9
    }).addTo(markersLayer);

    marker.bindPopup(`
      <div style="width:280px;">
        <b>${escapeHtml(lead.name||"")}</b><br/>
        <div style="color:rgba(234,240,255,.7); margin:6px 0;">${escapeHtml(lead.address||"")}</div>
        <div style="margin-top:8px;">
          <label style="display:block; font-size:12px; color:rgba(234,240,255,.7);">Labels</label>
          <input id="lbl_${lead.id}" value="${escapeAttr(lead.labels||"")}" style="width:100%; padding:9px 10px; border-radius:12px; border:1px solid rgba(255,255,255,.12); background:rgba(255,255,255,.03); color:#eaf0ff;">
        </div>
        <div style="margin-top:8px;">
          <label style="display:block; font-size:12px; color:rgba(234,240,255,.7);">Notes</label>
          <textarea id="nts_${lead.id}" style="width:100%; height:60px; padding:9px 10px; border-radius:12px; border:1px solid rgba(255,255,255,.12); background:rgba(255,255,255,.03); color:#eaf0ff;">${escapeHtml(lead.notes||"")}</textarea>
        </div>
        <button id="save_${lead.id}" style="margin-top:10px; width:100%; padding:10px 12px; border-radius:12px; border:1px solid rgba(76,141,255,.35); background:rgba(76,141,255,.18); color:#eaf0ff; font-weight:800; cursor:pointer;">Save</button>
      </div>
    `);

    marker.on("popupopen", ()=>{
      const btn = document.getElementById("save_"+lead.id);
      if(btn){
        btn.onclick = async ()=>{
          const lbl = document.getElementById("lbl_"+lead.id).value;
          const nts = document.getElementById("nts_"+lead.id).value;
          await updateRow(lead.id, lbl, nts);
          await loadAndRender(true);
        };
      }
    });

    marker.on("click", ()=>focusLead(lead.id, false));
    markersById.set(lead.id, marker);
  }
}

async function quickAddLabel(id, label){
  const lead = allLeads.find(x=>x.id===id);
  if(!lead) return;

  let labels = parseLabels(lead.labels);
  if(label === "Sent") labels = labels.filter(x=>x !== "Unsent");

  if(!labels.includes(label)) labels.push(label);

  await updateRow(id, labels.join(", "), lead.notes || "");
  await loadAndRender(true);
}

function focusLead(id, openPopup){
  const marker = markersById.get(id);
  if(marker){
    marker.bringToFront();
    map.setView(marker.getLatLng(), Math.max(map.getZoom(), 14), {animate:true});
    if(openPopup) marker.openPopup();
  }
}

function wireUI(){
  $("btnRefresh").onclick = ()=>loadAndRender(true);
  $("btnFit").onclick = ()=>{
    const leads = allLeads.filter(leadMatches).filter(l=>l.lat && l.lng);
    const b = leads.map(l=>[l.lat,l.lng]);
    if(b.length) map.fitBounds(b, {padding:[40,40]});
  };

  $("search").addEventListener("input", ()=>{
    activeSearch = $("search").value || "";
    loadAndRender(true);
  });

  $("labelFilter").addEventListener("change", ()=>{
    activeLabel = $("labelFilter").value || "";
    loadAndRender(true);
  });

  $("needsWebsite").addEventListener("change", ()=>{
    activeNeedsWebsite = $("needsWebsite").value || "";
    loadAndRender(true);
  });

  document.querySelectorAll("#chips .chip").forEach(ch=>{
    ch.addEventListener("click", ()=>{
      document.querySelectorAll("#chips .chip").forEach(x=>x.classList.remove("active"));
      ch.classList.add("active");
      activeChip = ch.dataset.chip || "";
      loadAndRender(true);
    });
  });

  document.querySelectorAll("#stats .stat").forEach(st=>{
    st.addEventListener("click", ()=>{
      document.querySelectorAll("#stats .stat").forEach(x=>x.classList.remove("active"));
      st.classList.add("active");
      activeChip = st.dataset.stat || "";
      document.querySelectorAll("#chips .chip").forEach(x=>x.classList.remove("active"));
      const target = document.querySelector(`#chips .chip[data-chip="${activeChip}"]`);
      if(target) target.classList.add("active");
      loadAndRender(true);
    });
  });
}

async function loadAndRender(force){
  try{
    const data = await fetchSheet();

    if(!force && lastRev && data.rev && data.rev === lastRev){
      $("status").textContent = "No changes (auto refresh 5s)";
      return;
    }
    lastRev = data.rev || null;

    const rows = data.rows || [];
    allLeads = rows.map(r=>{
      const lat = r["Lat"] ? parseFloat(r["Lat"]) : null;
      const lng = r["Lng"] ? parseFloat(r["Lng"]) : null;
      return {
        id: r.id,
        name: r[COL_NAME] || "",
        address: r[COL_ADDRESS] || "",
        area: r[COL_AREA] || "",
        phone: r[COL_PHONE] || "",
        website: r[COL_WEBSITE] || "",
        mapsUrl: r[COL_MAPS_URL] || "",
        labels: r[COL_LABELS] || "",
        notes: r[COL_NOTES] || "",
        rating: r[COL_RATING] || "",
        reviews: r[COL_REVIEWS] || "",
        needsWebsite: r[COL_NEEDS_WEBSITE] || "",
        lat, lng
      };
    });

    computeStats();
    rebuildLabelDropdown();

    const filtered = allLeads.filter(leadMatches);
    renderList(filtered);
    renderMarkers(filtered);

    $("status").textContent = `Loaded ${filtered.length}/${allLeads.length} leads • last rev: ${lastRev}`;
  }catch(e){
    $("status").textContent = "Error: " + (e.message || e);
    console.error(e);
  }
}

(function start(){
  initMap();
  wireUI();
  loadAndRender(true);
  setInterval(()=>loadAndRender(false), REFRESH_MS);
})();
